function [estpar,fval,out] = LWM_RND_core(M1,M2,oprice,strike, rf, TTM,Ft, weights,LWM_ini,LWM_options)
%==========================================================================================
%The function estimate the parameters of the LWM-based RND 
%based on Li, Nolte and Pham (2023).
%
%INPUT:
%   M1: non-negative integer scalar, number of Lognormal densities in the mixture
%   M2: non-negative integer scalar, number of Weibull densities in the mixture
%   oprice: N-by-1 call option prices
%   strike: N-by-1 strike prices of the options
%   rf: risk-free rate
%   TTM: time to maturity of the options (in years)
%   Ft: underlying futures price
%   Mmax: maximum number of mixtures for the LWM method. Default = 5
%   weights: N-by-1 weight vector. If not provided, assumed equal weights
%   LWM_ini: struct of initialization step. See LWM_RND_initialization.m
%   function. If not provided, used default.
%   LWM_options: options to fine-tune the LWM-based RND estimation. If not
%   provided, default options are used. See the function LWM_options_setup.m for
%   details.
%
%OUTPUT:
%   estpar: estimated parameter vector
%   fval: minimized objective function value
%   out: matrix of estpar and fval for upto Smax optimal solutions. 
%        See Smax in LWM_options_setup.m
%==========================================================================================
% This ver: 2023/05/24
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2023). Parametric Risk-Neutral 
%          Density Estimation via Finite Lognormal-Weibull Mixtures
%========================================================================================== 


warning('off','MATLAB:singularMatrix')

if nargin < 8
    weights = ones(size(oprice));
end

if nargin < 9
    LWM_ini = LWM_RND_initialization(oprice,strike, rf, TTM,Ft,weights);
end

if nargin < 10
    LWM_options = LWM_options_setup();
end

%total number of mixtures
M=M1 + M2;
%unpacking the options
Smax = LWM_options.Smax; 
Srun_max = LWM_options.Srun_max;
c = LWM_options.c; %variance constraint
Lmax = LWM_options.Lmax;
objval_scale=LWM_options.objval_scale;
Display=LWM_options.Display;

max_time = 20;%maximum allowed time per iteration in seconds


%unpacking the initialized values
kfit = LWM_ini.kfit;
sfit = LWM_ini.sfit;
fval_ub = min(min(LWM_ini.fvals,LWM_ini.fvalk)*objval_scale,LWM_options.fval_ub);
IV_atm = LWM_ini.IV_atm;
x0s=LWM_ini.x0s;
x0k=LWM_ini.x0k;



%setting of the fmincon optimization
TolX=1e-5; TolFun=1e-5;
fmincon_options = optimoptions(@fmincon,'Algorithm','interior-point',...
    'SpecifyObjectiveGradient',true,'SpecifyConstraintGradient',true,'ScaleProblem',false,...
    'OptimalityTolerance',TolFun,'StepTolerance',TolX,'checkgradient',false,...
    'MaxFunctionEvaluation',1e8,'MaxIter',1e8,'Display','off');

Sreset =1;
W = diag(weights);
out = [];
S=0;Srun = 1;%initializing the counters
if strcmp(Display,'final') || strcmp(Display,'iter')
   fprintf('Commencing (%d,%d)-lognormal-Weibull mixture estimation \n',M1,M2);
   t=tic;
end
fval_best = fval_ub;estpar_best = [];
while S<Smax && Srun<=Srun_max %run untill Smax optimal solutions are obtained or a total of Srun_max optimizations are computed
    %generating a starting value for each optimization of the NWLS
    LMoneyness = log(strike/Ft)/(IV_atm*sqrt(TTM));%log-monenyness of the option panel
    %generating a range of the mixture parameters F, which is the intersection
    %of the set [-3,1] and [min(LMoneyness), max(LMoneyness)]
    Srange = strike(LMoneyness>=max(-3, min(LMoneyness)) & LMoneyness<=min(1,max(LMoneyness)) );
    S_trials=zeros(3*M+1,Lmax);
    for L = 1:Lmax
        estpar0 = LWM_parm_extend(x0s,x0k,M1,M2,Srange,sfit,kfit);
        F0 = estpar0(M+1:2*M);
        %quadratic optimization to obtain feasible weights
        [~,~,~,~,oprices] = LWM_MSE(estpar0,M1,M2,oprice,strike, rf, TTM,[],objval_scale);
        H=oprices'*W*oprices;
        H=sparse((H+H')/2);
        f=-oprices'*W*oprice;
        Aeq=[ones(M,1)';F0'];
        beq = [1; Ft];
        lb=zeros(M,1);
        ub=ones(M,1);
        options = optimoptions('quadprog','Display','off');
        w_vec = quadprog(H,f,[],[],Aeq,beq,lb,ub,[],options);
        S_trials(:,L) = [[w_vec;estpar0(M+1:end)]; (oprices*w_vec-oprice)'*W*(oprices*w_vec-oprice)];
    end
    %a random starting point with optimal weights
    [~,m_ind]=min(S_trials(end,:));
    x0 = S_trials(1:end-1,m_ind);
    if isempty(estpar_best)
        estpar_best = x0;
    end
    
    
    %local NWLS optimization
    f=@(x) LWM_MSE(x,M1,M2,oprice,strike, rf, TTM,weights,objval_scale);
    fmincon_options.HessianFcn=@(x,l) LWM_hessFcn(x,l,M1,M2,oprice,strike, rf, TTM,weights,objval_scale);
    %equality constraint for the weights
    Aeq = [ones(1,M) zeros(1,2*M)]; beq =1;
    %inequality constraint for the variances
    Msize = M*(M-1);
    Avar = [];
    Mmat = eye(M);
    for L1= 2: M
        for L2 = 1:L1-1
            c_offest1= (L1<=M1)*sqrt(TTM*6)/pi+(L1>M1);
            c_offest2= (L2<=M1)*sqrt(TTM*6)/pi+(L1>M1);
            Avar(end+1,:) = [c*Mmat(L1,:)*c_offest1-Mmat(L2,:)*c_offest2];
            Avar(end+1,:) = [c*Mmat(L2,:)*c_offest2-Mmat(L1,:)*c_offest1];
        end
    end
    A = [zeros(Msize, 2*M) Avar]; b=zeros(Msize,1);
    %parameter bounds
    lb =[zeros(M,1);ones(M,1)*min(strike)*0.5;zeros(M,1)]; ub= [ones(M,1); ones(M,1)*max(strike)*1.5; ones(M,1)*100];
    %quadratic constraint for the mean of the LWM distribution
    nlcon = @(x) LWM_nlcon(x, Ft);
    tstart = tic;
    fmincon_options.OutputFcn = @(x,optim,state) LWM_timer(x,optim,state,tstart,max_time);
    [estpar_test,fval,flag] = fmincon(f,x0,A,b,Aeq,beq,lb,ub,nlcon,fmincon_options);
    %check the solution...
    if fval > fval_ub %optimized function must be smaller than the upper bound
        if strcmp(Display,'iter') 
            fprintf('fval (%.3f) larger than upper bound (%.3f), total optimization %d/%d  \n',fval,fval_ub,Srun,Srun_max);
        end
        Sreset = Sreset +1;
    elseif  flag ==1 %record the solution
        estpar=estpar_test;
        if fval<=fval_best
            estpar_best=estpar;
            fval_best=fval;
        end    
        S=S+1;
        out =[out [estpar; fval]];
        Sreset = 1;
        if strcmp(Display,'iter')
            fprintf('fval (%.3f) less than upper bound (%.3f), %d/%d completed,  total optimization %d/%d \n',fval,fval_ub,S,Smax,Srun,Srun_max);
        end
    else
         if strcmp(Display,'iter')
            fprintf('parameter constraint not satisfied,  total optimization %d/%d \n',Srun,Srun_max);
        end
    end
    Srun = Srun + 1;
end
if strcmp(Display,'iter') || strcmp(Display,'final')
   fprintf('(%d,%d)-lognormal-Weibull mixture estimation complete, time elapse %f \n',M1,M2,toc(t));
end
if strcmp(Display,'final')
    fprintf('%d/%d estimates computed, total optimization %d/%d \n',S,Smax,Srun-1,Srun_max);
end
if ~isempty(out)
estpar  = estpar_best; %parameter estimates
fval = fval_best; %estimated parameter value
else
    warning('No valid parameter estimates can be obtained within the maximum number of optimizations.')
    estpar = zeros(3*M,1);
    fval = Inf;
end
end

